#ifndef RawData_h_seen
#define RawData_h_seen

namespace gov {
	namespace cca {

/**
  A low-level interface for accessing memory.  This interface is designed
  principally for writers of tools rather than end users because it provides
  pointers directly to computer memory.  This interface may be an efficient
  way to copy data between components, but end users are normally encouraged
  to use higher-level interfaces or "views" on data.  An example would be one
  providing for array access patterns.

  Another use of this interface is to provide a wrapper for user defined
  types, as each element can be assigned a unique name.  In addition to the
  primitive data types (and pointers to them), a data element can be a pointer
  to a RawData object instance, which allows for use of the composite design
  pattern.
 */
class RawData : public virtual Port {
public:

  /** Name associated with this set of buffers.
   * @return the name. */
  virtual const char* getName() = 0;

  /** The total number of encapsulated buffers. 
   * @return nBuffers. */
  virtual int getNumberOfBuffers() = 0;

  /** Returns information about the buffer at the given index.  Buffers are
      indexed from 0 .. getNumberOfBuffers-1.
      @param bufferIndex Input: the index (from 0) of buffer requested.
      @param elementName Output: name associated with the buffer (default
                                 value is "_UNNAMED")
      @param elementType Output: enumerated type of the elements in the buffer.
      @param elementCount Output: length of the buffer.
      @param buffer Output: pointer to data buffer.
  */
  virtual void getBufferInfo(int bufferIndex, const char* & elementName,
                             enum RawDataType & elementType,
                             int & elementCount, void* & buffer) = 0;

  /** Returns the size of the given RawDataType in bytes. */
  virtual unsigned long sizeofDataType(enum RawDataType) = 0;

  /** An enum over C/Fortran primitives in multiple precisions. */
  enum Type {
    Err = 0,
    Byte = 1,
    Bool = 2,
    Char = 3, 
    WChar = 4,
    Int1 = 5, 
    Int2 = 6, 
    Int4 = 7, 
    Int8 = 8,
    UInt1 = 9, 
    UInt2 = 10, 
    UInt4 = 11, 
    UInt8 = 12,
    Float4 = 13, 
    Float8 = 14, 
    Float16 = 15,
    Complex4 = 16, 
    Complex8 = 17, 
    Complex16 = 18, 
    Complex32 = 19,
    PtrByte = 65,
    PtrBool = 66,
    PtrChar = 67, 
    PtrWChar = 68,
    PtrInt1 = 69, 
    PtrInt2 = 70, 
    PtrInt4 = 71, 
    PtrInt8 = 72,
    PtrUInt1 = 73, 
    PtrUInt2 = 74, 
    PtrUInt4 = 75, 
    PtrUInt8 = 76,
    PtrFloat4 = 77, 
    PtrFloat8 = 78, 
    PtrFloat16 = 79,
    PtrComplex4 = 80, 
    PtrComplex8 = 81, 
    PtrComplex16 = 82, 
    PtrComplex32 = 83,
    PtrRawData = 84
  };

}; // RawData
} ENDSEMI // cca
} ENDSEMI // gov

#endif // RawData_h_seen
