#ifndef KeyValueTyped_h_seen
#define KeyValueTyped_h_seen

namespace gov {
    namespace cca {
//! Abstract control parameters and other string named data io interface.
/** This is a generic properties input/output Port. It may be backed
    by a shared database, a private database, or an instance of some
    numerical object, or anything else for that matter.

    This is a Closed hash -- one value (and value type) per key.
    Setting a new value and type
    for a key which has a value of another type already is an error.

  As an example:
  If you have a solver that cares about the properties of
  a Matrix or Operator, simply :
      esi::Properties *p;
      err = mtx->getInterface("esi::Properties", (void *)p,msg);
      bool is_psym = false;
      if (err >= 0) {  // <0 means no properties interface p.
        err2 = p->getBool("esi.P-Symmetric",psym); // ignore err2 is ok
        // if esi.P-Symmetric unknown, nothing psym is returned unchanged.
      }

  Implementation notes:

    Since this version uses an enum-based type-checking scheme,
    access functions may return errors for data with types not
    supported on the given platform. It is probable that
    no 100% complete and native C compiler supported implementation
    will exist due to the breadth of possible types defined in 
    RawData.h

*/
class KeyValueTyped : public virtual Port {
public:
  /** Defines a new single-valued property or changes the value of an existing one.
      @param key The property name, which will be copied.
      @param type The singleton types from the Type enum in RawData.h. 
                  String values are supported but not with this function.
		  Ptr* types from RawData are not supported.
      @param value Pointer to the property value to be copied.
                   A 0 pointer value will cause the key to remain
		   with a NULL value. Use 'unset' to delete key.
      @return 0 if ok, 
      -3 if not ok because type is unsupported in the implementation.
      -2 if not ok due to a type conflict with existing value,
      and -1 for all other reason.
   */
  virtual int set(const char *key, enum RawData::Type type, void * value) = 0;

  /** 
   * @param key The name of the value to be fetched.
   * @param valuePtr The slot to fill with the value requested. (can't be 0).
   * @param type The singleton types from the Type enum in RawData.h, 
   *   per set() in this interface.
   * @return 0 if ok,
   *   -1 if key unknown.
   *   -2 if type conflict.
   */
  virtual int get(const char * key, enum RawData::Type type, void * valuePtr) = 0;

  /** Remove a key from the properties. All errors are silently ignored.
      @param pointerOut If the key corresponds to
      a void pointer property, the pointer is returned in pointerOut,
      otherwise pointerOut is 0.
   */
  virtual void unset(const char *propName) = 0;

  /** Defines a new string property or changes its value.
      @param key The property name, which will be copied.
      @param value The property value to be set.
      @return 0 if ok, -2 if not ok due to a type conflict,
      and -1 for all other reason.
   */
  virtual int setString(const char *key, const char *value) = 0;

  /** Fetch a string value.
   * @param key The name of the string value to be fetched.
   * @return 0 if ok,
   *   -1 if key unknown.
   *   -2 if type conflict (key is not for a string).
   */
  virtual const char * getString(const char * key) = 0;

  /** Produce a list of all the keys in an abstract container.
   * @param keylist input/output Argv that should be supplied empty
   * (containing no keys). The names of keys in the KeyValueTyped are appended.
   */
  virtual void getKeys(Argv * keylist) = 0;

  }; // keyvaluetyped
} ENDSEMI // cca
} ENDSEMI  // gov 

#endif // KeyValueTyped_h_seen

